<?php

namespace PhpOffice\PhpSpreadsheetTests\Writer\Html;

use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\RichText\RichText;
use PhpOffice\PhpSpreadsheet\Shared\File;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheetTests\Functional;

class XssVulnerabilityTest extends Functional\AbstractFunctional
{
    public function providerAcceptableMarkupRichText()
    {
        return [
            'basic text' => ['Hello, I am safely viewing your site', 'Hello, I am safely viewing your site'],
            'link' => ["<a href='Visit Google'>Google is here</a>", '<a href="Visit%20Google">Google is here</a>'],
        ];
    }

    /**
     * @dataProvider providerAcceptableMarkupRichText
     *
     * @param string $safeTextString
     * @param string $adjustedTextString
     */
    public function testMarkupInComment($safeTextString, $adjustedTextString): void
    {
        $spreadsheet = new Spreadsheet();

        $richText = new RichText();
        $richText->createText($safeTextString);

        $spreadsheet->getActiveSheet()->getCell('A1')->setValue('XSS Test');

        $spreadsheet->getActiveSheet()
            ->getComment('A1')
            ->setText($richText);

        $filename = tempnam(File::sysGetTempDir(), 'phpspreadsheet-test');

        $writer = IOFactory::createWriter($spreadsheet, 'Html');
        $writer->save($filename);

        $verify = file_get_contents($filename);
        unlink($filename);
        // Ensure that executable js has been stripped from the comments
        self::assertStringContainsString($adjustedTextString, $verify);
    }

    public function providerXssRichText()
    {
        return [
            'script tag' => ["Hello, I am trying to <script>alert('Hack');</script> your site"],
            'javascript tag' => ["<a href='&#x2000;javascript:alert(1)'>CLICK</a>"],
            'with unicode' => ['<a href="\\u0001java\\u0003script:alert(1)">CLICK<a>'],
            'inline css' => ['<li style="list-style-image: url(javascript:alert(0))">'],
            'char value chevron' => ["\x3cscript src=http://www.example.com/malicious-code.js\x3e\x3c/script\x3e"],
        ];
    }

    /**
     * @dataProvider providerXssRichText
     *
     * @param string $xssTextString
     */
    public function testXssInComment($xssTextString): void
    {
        $spreadsheet = new Spreadsheet();

        $richText = new RichText();
        $richText->createText($xssTextString);

        $spreadsheet->getActiveSheet()->getCell('A1')->setValue('XSS Test');

        $spreadsheet->getActiveSheet()
            ->getComment('A1')
            ->setText($richText);

        $filename = tempnam(File::sysGetTempDir(), 'phpspreadsheet-test');

        $writer = IOFactory::createWriter($spreadsheet, 'Html');
        $writer->save($filename);

        $verify = file_get_contents($filename);
        unlink($filename);
        // Ensure that executable js has been stripped from the comments
        self::assertStringNotContainsString($xssTextString, $verify);
    }
}
